// util.h
//
// Mark Johnson, (c) 8th November 2007, last modified (c) 2nd May 2012
//
// This is now compatible with the C++0x extensions to C++ (basically just the
// hash tables).  Use the g++ compiler flag -std=c++0x to use these extensions.
// If these aren't used, then this uses the hash tables from tr1.

#pragma once

#include <algorithm>
#include <cassert>
#include <cctype>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <ctime>
#include <functional>
#include <iostream>
#include <iterator>
#include <list>
#include <locale>
#include <map>
#include <numeric>
#include <set>
#include <sstream>
#include <string>
#include <utility>
#include <vector>

#include <sys/time.h>
#include <sys/resource.h>

#ifdef __GXX_EXPERIMENTAL_CXX0X__
  #include <unordered_map>
  #include <unordered_set>
  namespace stdhash=std;
  #define UTIL_BEGIN_STDHASHNAMESPACE namespace std 
  #define UTIL_END_STDHASHNAMESPACE 
#else  
  #include <tr1/unordered_map>
  #include <tr1/unordered_set>
  namespace stdhash=std::tr1;
  #define UTIL_BEGIN_STDHASHNAMESPACE namespace std { namespace tr1 
  #define UTIL_END_STDHASHNAMESPACE } 
#endif

// define some useful macros

#define HERE   __FILE__ << ":" << __LINE__ << ": In " << __func__ << "()"

// Uncomment this if you want the full function signature (despite the name
// it is hardly pretty!)
//
// #define HERE   __FILE__ << ":" << __LINE__ << ": In " << __PRETTY_FUNCTION__ 

#define TRACE1(expr)                                                         \
  std::cerr << HERE                                                          \
            << ", " << __STRING(expr) << " = " << (expr) << std::endl

#define TRACE2(expr1, expr2)						     \
  std::cerr << HERE                                                          \
            << ", " << __STRING(expr1) << " = " << (expr1)                   \
            << ", " << __STRING(expr2) << " = " << (expr2) << std::endl

#define TRACE3(expr1, expr2, expr3)					     \
  std::cerr << HERE                                                          \
            << ", " << __STRING(expr1) << " = " << (expr1)                   \
            << ", " << __STRING(expr2) << " = " << (expr2)                   \
            << ", " << __STRING(expr3) << " = " << (expr3) << std::endl

#define TRACE4(expr1, expr2, expr3, expr4)				     \
  std::cerr << HERE                                                          \
            << ", " << __STRING(expr1) << " = " << (expr1)                   \
            << ", " << __STRING(expr2) << " = " << (expr2)                   \
            << ", " << __STRING(expr3) << " = " << (expr3)                   \
            << ", " << __STRING(expr4) << " = " << (expr4) << std::endl

#define TRACE5(expr1, expr2, expr3, expr4, expr5)			     \
  std::cerr << HERE                                                          \
            << ", " << __STRING(expr1) << " = " << (expr1)                   \
            << ", " << __STRING(expr2) << " = " << (expr2)                   \
            << ", " << __STRING(expr3) << " = " << (expr3)                   \
            << ", " << __STRING(expr4) << " = " << (expr4)                   \
            << ", " << __STRING(expr5) << " = " << (expr5) << std::endl

#define TRACE6(expr1, expr2, expr3, expr4, expr5, expr6)		     \
  std::cerr << HERE                                                          \
            << ", " << __STRING(expr1) << " = " << (expr1)                   \
            << ", " << __STRING(expr2) << " = " << (expr2)                   \
            << ", " << __STRING(expr3) << " = " << (expr3)                   \
            << ", " << __STRING(expr4) << " = " << (expr4)                   \
            << ", " << __STRING(expr5) << " = " << (expr5)                   \
            << ", " << __STRING(expr6) << " = " << (expr6) << std::endl

#define TRACE7(expr1, expr2, expr3, expr4, expr5, expr6, expr7)		     \
  std::cerr << HERE                                                          \
            << ", " << __STRING(expr1) << " = " << (expr1)                   \
            << ", " << __STRING(expr2) << " = " << (expr2)                   \
            << ", " << __STRING(expr3) << " = " << (expr3)                   \
            << ", " << __STRING(expr4) << " = " << (expr4)                   \
            << ", " << __STRING(expr5) << " = " << (expr5)                   \
            << ", " << __STRING(expr6) << " = " << (expr6)                   \
            << ", " << __STRING(expr7) << " = " << (expr7) << std::endl

// ASSERT() is a version of assert() that is always checked, no matter what
// NDEBUG is set to
//
#define ASSERT(expr) { if (!(expr)) std::cerr << HERE << ", assertion \"" << __STRING(expr) << "\" failed." << util::abort; }

///////////////////////////////////////////////////////////////////////////
//                                                                       //
//                              Looping constructs                       //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

//! foreach is a simple loop construct
//!
//!   STORE should be an STL container
//!   TYPE is the typename of STORE
//!   VAR will be defined as a local variable of type TYPE::iterator
//
#define foreach(TYPE, VAR, STORE) \
   for (TYPE::iterator VAR = (STORE).begin(); VAR != (STORE).end(); ++VAR)

//! cforeach is just like foreach, except that VAR is a const_iterator
//!
//!   STORE should be an STL container
//!   TYPE is the typename of STORE
//!   VAR will be defined as a local variable of type TYPE::const_iterator
//
#define cforeach(TYPE, VAR, STORE) \
   for (TYPE::const_iterator VAR = (STORE).begin(); VAR != (STORE).end(); ++VAR)



namespace util {

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                             Map searching                             //
  //                                                                       //
  // dfind(map, key) returns the key's value in map, or map's default      //
  //   value if no such key exists (the default value is not inserted)     //
  //                                                                       //
  // afind(map, key) returns a reference to the key's value in map, and    //
  //    asserts that this value exists                                     //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! dfind(Map, Key) returns the value Map associates with Key, or the
  //!  Map's default value if no such Key exists
  //
  template <class Map, class Key>
  inline typename Map::mapped_type dfind(const Map& m, const Key& k)
  {
    typename Map::const_iterator i = m.find(k);
    if (i == m.end())
      return typename Map::mapped_type();
    else
      return i->second;
  }  // util::dfind()

  //! dfind(Map, Key, Default) returns the value Map associates with Key, or 
  //!   Default if no such Key exists
  //
  template <typename Map, typename Key, typename Value>
  inline typename Map::mapped_type dfind(const Map& m, const Key& k, const Value& d)
  {
    typename Map::const_iterator i = m.find(k);
    if (i == m.end())
      return d;
    else
      return i->second;
  }  // util::dfind()

  //! afind(map, key) returns a reference to the value associated
  //!  with key in map.  It uses assert to check that the key's value
  //!  is defined.
  //
  template <class Map, class Key>
  inline typename Map::mapped_type& afind(Map& m, const Key& k)
  {
    typename Map::iterator i = m.find(k);
    assert(i != m.end());
    return i->second;
  }  // util::afind()

  template <class Map, class Key>
  inline const typename Map::mapped_type& afind(const Map& m, const Key& k)
  {
    typename Map::const_iterator i = m.find(k);
    assert(i != m.end());
    return i->second;
  }  // util::afind()

  //! incr() increments the value associated with key in map, deleting the key,value
  //! pair if the incremented value is zero.
  //
  template <typename Map, typename Key, typename Inc>
  inline typename Map::mapped_type incr(Map& m, const Key& k, Inc i) {
    std::pair<typename Map::iterator, bool> itb = m.insert(typename Map::value_type(k,i));
    if (itb.second==false && (itb.first->second += i) == typename Map::mapped_type()) {
      m.erase(itb.first);
      return typename Map::mapped_type();
    }
    else
      return itb.first->second;    
  }  // util::incr()

  template <typename Map, typename Key>
  inline typename Map::mapped_type incr(Map& m, const Key& k) { return incr(m, k, 1); }

  //! insert_newkey(map, key, value) checks that map does not contain
  //! key, and binds key to value.
  //
  template <class Map, class Key, class Value>
  inline typename Map::value_type& 
  insert_newkey(Map& m, const Key& k,const Value& v) 
  {
    std::pair<typename Map::iterator, bool> itb 
      = m.insert(Map::value_type(k, v));
    assert(itb.second);
    return *(itb.first);
  }  // util::insert_newkey()

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                  insert and increment iterators                       //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! An assoc_insert_iterator inserts an object into an associative container.
  //! This implementation is based on the Josuttis "The C++ Standard Library", p 289.
  //
  template <typename Container>
  class assoc_insert_iterator : public std::iterator<std::output_iterator_tag,void,void,void,void> 
  {
  protected:
    Container& container;           //!< container into which objects are inserted
    
  public:
    explicit assoc_insert_iterator(Container& c) : container(c) { }

    //! operator= inserts value into container
    //
    assoc_insert_iterator<Container>& 
    operator= (const typename Container::value_type& value) {
      container.insert(value);
      return *this;
    }

    //! operator* is a no-op that returns the iterator
    //
    assoc_insert_iterator<Container>& operator* () { return *this; }

    //! operator++ is a no-op that returns the iterator
    //
    assoc_insert_iterator<Container>& operator++ () { return *this; }

    //! operator++ is a no-op that returns the iterator
    //
    assoc_insert_iterator<Container>& operator++ (int) { return *this; }
    
  };  // util::assoc_insert_iterator{}

  //! inserter() selects the right kind of insert_iterator for this container
  //
  template <typename Key, typename Value> inline assoc_insert_iterator<stdhash::unordered_map<Key,Value> > 
  inserter(stdhash::unordered_map<Key,Value>& c) {
    return assoc_insert_iterator<stdhash::unordered_map<Key,Value> >(c);
  }  // util::inserter()

  template <typename Key, typename Value> inline assoc_insert_iterator<std::map<Key,Value> > 
  inserter(std::map<Key,Value>& c) {
    return assoc_insert_iterator<std::map<Key,Value> >(c);
  }  // util::inserter()

  template <typename X> inline std::back_insert_iterator<std::vector<X> >
  inserter(std::vector<X>& c) {
    return std::back_insert_iterator<std::vector<X> >(c);
  }  // util::inserter()

  //! An assoc_increment_iterator increments the count associated with a key by 1
  //
  template <typename Container>
  class assoc_increment_iterator : public std::iterator<std::output_iterator_tag,void,void,void,void> 
  {
  protected:
    Container& container;           //!< container into which objects are inserted
    
  public:
    explicit assoc_increment_iterator(Container& c) : container(c) { }

    //! operator= inserts value into container
    //
    assoc_increment_iterator<Container>& 
    operator= (const typename Container::key_type& key) {
      incr(container, key);
      return *this;
    }

    //! operator* is a no-op that returns the iterator
    //
    assoc_increment_iterator<Container>& operator* () { return *this; }

    //! operator++ is a no-op that returns the iterator
    //
    assoc_increment_iterator<Container>& operator++ () { return *this; }

    //! operator++ is a no-op that returns the iterator
    //
    assoc_increment_iterator<Container>& operator++ (int) { return *this; }
    
  };  // util::assoc_increment_iterator{}

  //! incrementer() selects the right kind of increment_iterator for this container
  //
  template <typename Key, typename Value> inline assoc_increment_iterator<stdhash::unordered_map<Key,Value> > 
  incrementer(stdhash::unordered_map<Key,Value>& c) {
    return assoc_increment_iterator<stdhash::unordered_map<Key,Value> >(c);
  }  // util::incrementer()

  template <typename Key, typename Value> inline assoc_increment_iterator<std::map<Key,Value> > 
  incrementer(std::map<Key,Value>& c) {
    return assoc_increment_iterator<std::map<Key,Value> >(c);
  }  // util::incrementer()

  
  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //               simplified interface to standard functions              //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! find() returns an iterator to an element in a sequence
  //
  template <typename Xs, typename X> inline typename Xs::iterator
  find(Xs& xs, const X& x) {
    return std::find(xs.begin(), xs.end(), x);
  }  // util::find()

  template <typename Xs, typename X> inline typename Xs::const_iterator
  find(const Xs& xs, const X& x) {
    return std::find(xs.begin(), xs.end(), x);
  }  // util::find()

  //! sum() returns the sum of elements in a container
  //
  template <typename Xs> inline typename Xs::value_type 
  sum(const Xs& xs) {
    typedef typename Xs::value_type X;
    return std::accumulate(xs.begin(), xs.end(), X(), std::plus<X>());
  }  // util::sum()

  //! sum_second() returns the sum of the second components of a container (e.g., a map)
  //
  template <typename XYs> inline typename XYs::value_type::second_type
  sum_second(const XYs& xys) {
    typename XYs::value_type::second_type sum=0;
    cforeach (typename XYs, it, xys)
      sum += it->second;
    return sum;
  }  // util::sum_second()

  //! copy() appends all of the elements in Xs onto the end of Ys
  //
  template <typename Xs, typename Ys> void
  copy(const Xs& xs, Ys& ys) {
    std::copy(xs.begin(), xs.end(), inserter(ys));
  }  // util::copy()

  //! sort() sorts all of the elements in a container
  //
  template <typename Xs> inline void
  sort(Xs& xs) {
    std::sort(xs.begin(), xs.end());
  }  // util::sort()

  //! sort() sorts all of the elements in a container ordered by comp
  //
  template <typename Xs, typename Comp> inline void
  sort(Xs& xs, Comp comp) {
    std::sort(xs.begin(), xs.end(), comp);
  }  // util::sort()

  //! partial_sum() replaces each element in a container with the sum of its
  //! value and all preceding elements.  This is the first step in selecting
  //! an element from an urn.
  //
  template <typename Vs> inline void
  partial_sum(Vs& vs) {
    std::partial_sum(vs.begin(), vs.end(), vs.begin());
  }

  //! lower_bound() returns the largest index i such that vs[i] <= bound.
  //! This is the second step in selecting an element from an urn.
  //
  template <typename Vs, typename V> inline unsigned int
  lower_bound(const Vs& vs, const V bound) {
    typename Vs::const_iterator it = std::lower_bound(vs.begin(), vs.end(), bound);
    return it - vs.begin();
  }

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                       comparison functions                            //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! first_lessthan()(e1, e2) iff e1.first < e2.first
  //
  struct first_lessthan {
    template <typename T1, typename T2>
    bool operator() (const T1& e1, const T2& e2) const {
      return e1.first < e2.first;
    }
  };

  struct second_lessthan {
    template <typename T1, typename T2>
    bool operator() (const T1& e1, const T2& e2) const {
      return e1.second < e2.second;
    }
  };

  struct first_greaterthan {
    template <typename T1, typename T2>
    bool operator() (const T1& e1, const T2& e2) const {
      return e1.first > e2.first;
    }
  };

  struct second_greaterthan {
    template <typename T1, typename T2>
    bool operator() (const T1& e1, const T2& e2) const {
      return e1.second > e2.second;
    }
  };

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                          useful functions                             //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! trim_capacity() reallocates a vector to contain just the allocated elements
  //
  template <typename X>
  inline void trim_capacity(std::vector<X>& xs) {
    std::vector<X>(xs).swap(xs);
  }  // util::trim_capacity()

  //! randseed() returns an integer based on the time, to be used as a random number seed
  //
  inline unsigned int randseed() {
    time_t t;
    return time(&t);
  }  // util::randseed()

  //! split() parses a string by repeatedly reading a value of type Xs::value_type from it,
  //! calling xform() on that value, and pushing the result onto xs.  
  //! It returns the number of characters read from the string.
  //
  template <typename CharT, typename Traits, typename Alloc, typename Xform, typename Xs>
  inline std::ios::pos_type 
  split(const std::basic_string<CharT,Traits,Alloc>& str, Xform& xform, Xs& xs) {
    std::basic_istringstream<CharT,Traits> is(str);
    std::basic_string<CharT,Traits> x;
    while (is >> x)
      xs.push_back(xform(x));
    return is.tellg();
  }  // util::split()

  //! split() parses a string by repeatedly reading a value of type Xs::value_type from it and
  //! pushing it onto xs.  It returns the number of characters read from the string.
  //
  template <typename CharT, typename Traits, typename Alloc, typename Xs>
  inline std::ios::pos_type 
  split(const std::basic_string<CharT,Traits,Alloc>& str, Xs& xs) {
    std::basic_istringstream<CharT,Traits> is(str);
    typename Xs::value_type x;
    while (is >> x)
      xs.push_back(x);
    return is.tellg();
  }  // util::split()

  //! This version of split() returns a vector of strings obtained by splitting
  //! str at blanks.  It asserts that the string is completely analysed.
  //
  template <typename CharT, typename Traits, typename Alloc>
  inline std::vector<std::basic_string<CharT,Traits,Alloc> >
  split(const std::basic_string<CharT,Traits,Alloc>& str) {
    std::vector<std::basic_string<CharT,Traits,Alloc> > xs;
    std::ios::pos_type pos = split(str, xs);
    assert(pos == -1);
    return xs;
  }  // util::split()

  //! tolower() destructively transforms a string to a lower-case variant
  //
  template <typename CharT, typename Traits, typename Alloc> inline 
  std::basic_string<CharT,Traits,Alloc>& 
  tolower(std::basic_string<CharT,Traits,Alloc>& s) {
    std::locale loc;
    for (typename std::basic_string<CharT,Traits,Alloc>::iterator it = s.begin(); it != s.end(); ++it)
      *it = std::use_facet<std::ctype<CharT> >(loc).tolower(*it);
    return s;
  }  // util::tolower()

  //! lowercase() returns a lowercased copy of a string
  //
  template <typename CharT, typename Traits, typename Alloc> inline 
  std::basic_string<CharT,Traits,Alloc> 
  lowercase(std::basic_string<CharT,Traits,Alloc> s) {
    return util::tolower(s);
  }  // util::lowercase()

  //! all_whitespace() is true iff string is only composed of whitespace chars
  //
  template <typename CharT, typename Traits, typename Alloc> inline bool
  all_whitespace(std::basic_string<CharT,Traits,Alloc> s) {
    typedef std::basic_string<CharT,Traits,Alloc> S;
    std::locale loc;
    for (typename S::const_iterator it = s.begin(); it != s.end(); ++it)
      if (! std::use_facet<std::ctype<CharT> >(loc).is(std::ctype<char>::space, *it))
	return false;
    return true;
  }   // util::all_whitespace()


  //! normalise() sets ys[i] to xs[i]/s, where s is the sum of the xs,
  //!  and returns s.
  //
  template <typename Xs, typename Ys> inline typename Ys::value_type
  normalise(const Xs& xs, Ys& ys) {
    assert(xs.size() == ys.size());
    typedef typename Ys::value_type Y;
    Y s = sum(xs);
    assert(s > 0);
    std::transform(xs.begin(), xs.end(), ys.begin(), std::bind2nd(std::divides<Y>(), s));
    return s;
  }  // util::normalise()

  //! normalize() divides each element of xs by its sum s, which it returns.
  //
  template <typename Xs> inline typename Xs::value_type
  normalise(Xs& xs) {
    return normalise(xs, xs);
  }  // util::normalise()

  //! fill() is a more convenient interface to std::fill, which
  //!  replaces all of the elements in container with the specified value
  //
  template <typename Xs, typename X> inline void
  fill(Xs& xs, const X& x) {
    std::fill(xs.begin(), xs.end(), x);
  }  // util::fill()

  //! rank_elements_cmp{} is a function object used by rank_elements.
  //
  template <typename Xs, typename Cmp> 
  struct rank_elements_cmp {
    const Xs& xs;
    const Cmp& cmp;
    rank_elements_cmp(const Xs& xs, const Cmp& cmp) : xs(xs), cmp(cmp) { }
    template <typename I, typename J> bool operator() (I i, J j) const { return cmp(xs[i], xs[j]); }
  };  // rank_elements_cmp{}

  //! rank_elements() sets rs such that xs[rs[i]] <= xs[rs[i+1]], 
  //!  i.e., rs is a permutation such that xs is in sorted order.
  //!  rs should be of some suitable integer type.
  //
  template <typename Xs, typename Rs, typename Cmp> inline void
  rank_elements(const Xs& xs, Rs& rs, const Cmp& cmp) {
    rs.resize(xs.size());
    for (unsigned i = 0; i < rs.size(); ++i)
      rs[i] = i;
    sort(rs, rank_elements_cmp<Xs, Cmp>(xs, cmp));
  }   // rank_elements()

  template <typename Xs, typename Rs> inline void
  rank_elements(const Xs& xs, Rs& rs) {
    rs.resize(xs.size());
    for (unsigned i = 0; i < rs.size(); ++i)
      rs[i] = i;
    typedef typename Xs::value_type X;
    sort(rs, rank_elements_cmp<Xs, std::less<X> >(xs, std::less<X>()));
  }   // rank_elements()

  template <typename Float>
  inline Float abs(Float x) { return (x >= 0) ? x : -x; }

  template <typename X, typename Y>
  inline X min(X x, Y y) {
    if (y < x)
      return y;
    else
      return x;
  }

  template <typename X, typename Y>
  inline X max(X x, Y y) {
    if (x < y)
      return y;
    else
      return x;
  }

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                          useful predicates                            //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! approx_equal() is true of two real numbers if their relative difference
  //! is less than rtol
  //
  inline bool approx_equal(double x1, double x2, double rtol=1e-7) {
    return (2*fabs(x1 - x2) < rtol * (fabs(x1) + fabs(x2)));
  }

  //! assert_approx_equal() throws an error if the relative difference between
  //! x1 and x2 is greater than 1e-7.
  //
#ifdef NDEBUG
#define assert_approx_equal(x1, x2)
#else
#define assert_approx_equal(x1, x2)                                          \
  { if (!util::approx_equal(x1,x2))					     \
      std::cerr << HERE                                                      \
                << ", assert_approx_equal failed, "                          \
                << __STRING(x1) <<  " = " << (x1)                            \
                << " !~= " << __STRING(x2) << " = " << x2	             \
                << util::abort; }
#endif

  //! equal_contents() is true of two unordered_maps if their sorted contents
  //! are the same
  //
  template <typename K, typename V>
  bool equal_contents(const stdhash::unordered_map<K,V>& m1, const stdhash::unordered_map<K,V>& m2) {
    if (m1.size() != m2.size())   // quick failure test
      return false;
    typedef std::pair<K,V> KV;
    typedef std::vector<KV> KVs;
    KVs kvs1, kvs2;
    std::copy(m1.begin(), m1.end(), std::back_inserter(kvs1));
    assert(m1.size() == kvs1.size());
    std::sort(kvs1.begin(), kvs1.end());
    std::copy(m2.begin(), m2.end(), std::back_inserter(kvs2));
    assert(m2.size() == kvs2.size());
    std::sort(kvs2.begin(), kvs2.end());
    return kvs1 == kvs2;
  }

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                          IO stream functions                          //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////
  
  //! abort() causes the program to abort
  //
  inline std::ostream& abort(std::ostream& os) {
    os << std::endl;
    std::abort();
    return os;
  }  // util::abort()

  //! exit_failure() causes the program to exit with failure
  //
  inline std::ostream& exit_failure(std::ostream& os) {
    os << std::endl;
    std::exit(EXIT_FAILURE);
    return os;
  }  // util::exit_failure()

  //! exit_success() causes the program to exit normally
  //
  inline std::ostream& exit_success(std::ostream& os) {
    os << std::endl;
    std::exit(EXIT_SUCCESS);
    return os;
  }  // util::exit_failure()

  //! date() prints the current date to the stream
  //
  inline std::ostream& date(std::ostream& os) {
    time_t t;
    time(&t);
    return os << ctime(&t);
  }  // util::date()

  //! runtime() returns the user run time for this process
  //
  inline double runtime(int who=RUSAGE_SELF) {
    struct rusage ru;
    getrusage(who, &ru);
    return ru.ru_utime.tv_sec + ru.ru_utime.tv_usec/1.0e6;
  }

  //! resource_usage() prints out resource usage information
  //
  inline std::ostream& resource_usage(std::ostream& os) {
    FILE* fp = fopen("/proc/self/stat", "r");
    if (!fp) 
      return os << "runtime = " << runtime() << ", no /proc/self/stat";
    int utime;
    int stime;
    unsigned int vsize;
    unsigned int rss;
    int result = 
      fscanf(fp, "%*d %*s %*c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u %d %d %*d %*d %*d %*d"
	     "%*u %*u %*d %u %u", &utime, &stime, &vsize, &rss);
    ASSERT(result == 4);
    fclose(fp);
    // s << "utime = " << utime << ", stime = " << stime << ", vsize = " << vsize << ", rss = " << rss;
    // return s << "utime = " << utime << ", vsize = " << vsize;
    return os << "utime " << float(utime)/1.0e2 << "s, vsize " 
	      << float(vsize)/1048576.0 << " Mb.";
  }  // util::resource_usage()

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                             subsequence                               //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //!< A subsequence is a contiguous subsequence of another sequence.
  //!< Iterator should be an iterator into a sequence.
  //
  template <typename Iterator>
  struct subsequence {
    typedef Iterator iterator;
    typedef Iterator const_iterator;
    typedef typename std::iterator_traits<Iterator> value_type;

    const Iterator first;   //!< value of begin()
    const Iterator second;  //!< value of end()

    subsequence(Iterator first, Iterator second) : first(first), second(second) { }
    subsequence(Iterator start, size_t size) : first(start) { std::advance(start, size); second=start; }

    value_type& operator[] (size_t offset) { Iterator it(first); std::advance(it,offset); return *it; }
    const value_type& operator[] (size_t offset) const { Iterator it(first); std::advance(it,offset); return *it; }

    iterator begin() { return first; }
    iterator end() { return second; }

    const_iterator begin() const { return first; }
    const_iterator end() const { return second; }

    size_t size() const { return std::distance(first, second); }

    template <typename Container>
    bool operator== (const Container& c) const {
      iterator it0  = begin();
      typename Container::const_iterator it1 = c.begin();
      while (it0 != end()) 
	if (it1 == c.end() || *it0++ != *it1++)
	  return false;
      return it1 == c.end();
    }  // util::subsequence::operator== ()

    template <typename Container>
    bool operator!= (const Container& c) const {
      return ! operator==(c);
    }  // util::subsequence::operator!= ()

    template <typename Container>
    bool operator< (const Container& c) const { 
      return std::lexicographical_compare(begin(), end(), c.begin(), c.end()); 
    }  // util::subsequence::operator< ()

    template <typename Container>
    bool operator> (const Container& c) const { 
      return std::lexicographical_compare(c.begin(), c.end(), begin(), end()); 
    }  // util::subsequence::operator< ()

    //! hash() computes a hash function for the container
    //
    std::size_t hash() const {
      std::size_t h = 0; 
      std::size_t g;
      iterator p = begin();
      while (p != end()) {
	h = (h << 4) + (*p++);
	if ((g = h&0xf0000000)) {
	  h = h ^ (g >> 24);
	  h = h ^ g;
	}}
      return h;
    }  // util::subsequence::hash()

  };  // util::subsequence{}
  
}  // namespace util


///////////////////////////////////////////////////////////////////////////
//                                                                       //
//                             Hash functions                            //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

UTIL_BEGIN_STDHASHNAMESPACE {

  //! hash function for arbitrary pairs.  This is actually not such a great hash;
  //! particularly if the pairs are used to build arbitrary trees.
  //
  template <typename T1, typename T2> struct hash<std::pair<T1,T2> >
    : public std::unary_function<std::pair<T1,T2>, std::size_t> {
    std::size_t operator() (const std::pair<T1,T2>& p) const {
      std::size_t h1 = hash<T1>()(p.first);
      std::size_t h2 = hash<T2>()(p.second);
      return h1 ^ (h1 << 7) ^ h2 ^ (h2 << 5);
    }
  };  // stdhash::hash<std::pair<T1,T2> >

  //! hash function for arbitrary vectors.
  //
  template <typename X> struct hash<std::vector<X> >
    : public std::unary_function<std::vector<X>, std::size_t> {
    
    std::size_t operator() (const std::vector<X>& xs) const {
      std::size_t h = 0; 
      std::size_t g;
      typename std::vector<X>::const_iterator p = xs.begin();
      while (p != xs.end()) {
	h = (h << 4) + stdhash::hash<X>()(*p++);
	if ((g = h&0xf0000000)) {
	  h = h ^ (g >> 24);
	  h = h ^ g;
	}}
      return h;
    }  // stdhash::hash<std::vector<X> >::operator()

  }; // stdhash::hash<std::vector<X> >

  //! hash function for subsequences.  Redirects to the hash function in subsequence{}
  //
  template <typename Iterator> struct hash<util::subsequence<Iterator> >
    : public std::unary_function<util::subsequence<Iterator>, std::size_t> {
    std::size_t operator() (const util::subsequence<Iterator>& s) const { return s.hash(); }
  };  // stdhash::hash<util::subsequence>{}


} UTIL_END_STDHASHNAMESPACE // namespace stdhash

///////////////////////////////////////////////////////////////////////////
//                                                                       //
//                          Input/Output functions                       //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

// Pairs
//
template <typename X, typename Y>
std::ostream& operator<< (std::ostream& os, const std::pair<X,Y>& xy) {
  return os << '(' << xy.first << ' ' << xy.second << ')';
}

template <class T1, class T2>
std::istream& operator>> (std::istream& is, std::pair<T1,T2>& p)
{
  char c;
  if (is >> c) {
    if (c == '(') {
      if (is >> p.first >> p.second >> c && c == ')')
	return is;
      else 
	is.setstate(std::ios::badbit);
    }
    else
      is.putback(c);
  }
  is.setstate(std::ios::failbit);
  return is;
}

// Lists
//
template <class T>
std::ostream& operator<< (std::ostream& os, const std::list<T>& xs)
{
  os << '(';
  for (typename std::list<T>::const_iterator xi = xs.begin(); xi != xs.end(); ++xi) {
    if (xi != xs.begin())
      os << ' ';
    os << *xi;
  }
  return os << ')';
}

template <class T>
std::istream& operator>> (std::istream& is, std::list<T>& xs)
{
  char c;                          // This code avoids unnecessary copy
  if (is >> c) {                   // read the initial '('
    if (c == '(') {
      xs.clear();                  // clear the list
      do {
	xs.push_back(T());         // create a new elt in list
	is >> xs.back();           // read element
      }
      while (is.good());           // read as long as possible
      xs.pop_back();               // last read failed; pop last elt
      is.clear(is.rdstate() & ~std::ios::failbit);  // clear failbit
      if (is >> c && c == ')')     // read terminating ')'
	return is;                 // successful return
      else 
	is.setstate(std::ios::badbit); // something went wrong, set badbit
    }
    else                           // c is not '('
      is.putback(c);               //  put c back into input
  }
  is.setstate(std::ios::failbit);  // read failed, set failbit
  return is;
}

// Vectors
//
template <class T>
std::ostream& operator<< (std::ostream& os, const std::vector<T>& xs)
{
  os << '(';
  for (typename std::vector<T>::const_iterator xi = xs.begin(); xi != xs.end(); ++xi) {
    if (xi != xs.begin())
      os << ' ';
    os << *xi;
  }
  return os << ')';
}

template <class T>
std::istream& operator>> (std::istream& is, std::vector<T>& xs)
{
  char c;                          // This code avoids unnecessary copy
  if (is >> c) {                   // read the initial '('
    if (c == '(') {
      xs.clear();                  // clear the list
      do {
	xs.push_back(T());         // create a new elt in list
	is >> xs.back();           // read element
      }
      while (is.good());           // read as long as possible
      xs.pop_back();               // last read failed; pop last elt
      is.clear(is.rdstate() & ~std::ios::failbit);  // clear failbit
      if (is >> c && c == ')')     // read terminating ')'
	return is;                 // successful return
      else 
	is.setstate(std::ios::badbit); // something went wrong, set badbit
    }
    else                           // c is not '('
      is.putback(c);               //  put c back into input
  }
  is.setstate(std::ios::failbit);  // read failed, set failbit
  return is;
}

// Maps
//
template <typename Key, typename Value>
std::ostream& operator<< (std::ostream& os, const std::map<Key,Value>& k_v) {
  os << '(';
  const char* sep = "";
  for (typename std::map<Key,Value>::const_iterator it=k_v.begin(); it!=k_v.end(); ++it) {
    os << sep << it->first << '=' << it->second;
    sep = ", ";
  }
  return os << ')';
}

template <class Key, class Value>
std::istream& operator>> (std::istream& is, std::map<Key,Value>& m)
{
  char c;
  if (is >> c) {
    if (c == '(') {
      m.clear();
      std::pair<Key,Value> e;
      while (is >> e)
	m.insert(e);
      is.clear(is.rdstate() & ~std::ios::failbit);
      if (is >> c && c == ')')
	return is;
      else
	is.setstate(std::ios::badbit);
    }
    else
      is.putback(c);
  }
  is.setstate(std::ios::failbit);
  return is;
}

// Unorderd maps
//
template <typename Key, typename Value>
std::ostream& operator<< (std::ostream& os, const stdhash::unordered_map<Key,Value>& k_v) {
  os << '(';
  const char* sep = "";
  for (typename stdhash::unordered_map<Key,Value>::const_iterator it=k_v.begin(); it!=k_v.end(); ++it) {
    os << sep << it->first << '=' << it->second;
    sep = ", ";
  }
  return os << ')';
}

template <class Key, class Value>
std::istream& operator>> (std::istream& is, stdhash::unordered_map<Key,Value>& m)
{
  char c;
  if (is >> c) {
    if (c == '(') {
      m.clear();
      std::pair<Key,Value> e;
      while (is >> e)
	m.insert(e);
      is.clear(is.rdstate() & ~std::ios::failbit);
      if (is >> c && c == ')')
	return is;
      else
	is.setstate(std::ios::badbit);
    }
    else
      is.putback(c);
  }
  is.setstate(std::ios::failbit);
  return is;
}

// Sets
//
template <typename Value>
std::ostream& operator<< (std::ostream& os, const std::set<Value>& vs) {
  os << '(';
  const char* sep = "";
  for (typename std::set<Value>::const_iterator it = vs.begin(); it != vs.end(); ++it) {
    os << sep << *it;
    sep = " ";
  }
  return os << ')';
}

template <class T>
std::istream& operator>> (std::istream& is, std::set<T>& s)
{
  char c;
  if (is >> c) {
    if (c == '(') {
      s.clear();
      T e;
      while (is >> e)
	s.insert(e);
      is.clear(is.rdstate() & ~std::ios::failbit);
      if (is >> c && c == ')')
	return is;
      else
	is.setstate(std::ios::badbit);
    }
    else
      is.putback(c);
  }
  is.setstate(std::ios::failbit);
  return is;
}

// Unordered sets
//
template <typename Value>
std::ostream& operator<< (std::ostream& os, const stdhash::unordered_set<Value>& vs) {
  os << '(';
  const char* sep = "";
  for (typename stdhash::unordered_set<Value>::const_iterator it = vs.begin(); it != vs.end(); ++it) {
    os << sep << *it;
    sep = " ";
  }
  return os << ')';
}

template <class T>
std::istream& operator>> (std::istream& is, stdhash::unordered_set<T>& s)
{
  char c;
  if (is >> c) {
    if (c == '(') {
      s.clear();
      T e;
      while (is >> e)
	s.insert(e);
      is.clear(is.rdstate() & ~std::ios::failbit);
      if (is >> c && c == ')')
	return is;
      else
	is.setstate(std::ios::badbit);
    }
    else
      is.putback(c);
  }
  is.setstate(std::ios::failbit);
  return is;
}


template <typename Iterator> 
std::ostream& operator<< (std::ostream& os, const util::subsequence<Iterator>& ss) {
  cforeach(typename util::subsequence<Iterator>, it, ss)
    os << *it;
  return os;
}


///////////////////////////////////////////////////////////////////////////
//                                                                       //
//                            Input functions                            //
//                                                                       //
///////////////////////////////////////////////////////////////////////////

//! istream >> const char* consumes the characters from the istream.  
//! Just as in scanf, a space consumes an arbitrary amount of whitespace.
//
inline std::istream& operator>> (std::istream& is, const char* cp)
{
  if (*cp == '\0')
    return is;
  else if (*cp == ' ') {
    char c;
    if (is.get(c)) {
      if (isspace(c))
	return is >> cp;
      else {
	is.unget();
	return is >> (cp+1);
      }
    }
    else {
      is.clear(is.rdstate() & ~std::ios::failbit);  // clear failbit
      return is >> (cp+1);
    }
  }
  else {
    char c;
    if (is.get(c)) {
      if (c == *cp)
	return is >> (cp+1);
      else {
	is.unget();
	is.setstate(std::ios::failbit);
      }
    }
    return is;
  }
}

namespace util {

  //! A default_value_type{} object is used to read an object from a stream,
  //! assigning a default value if the read fails.  Users should not need to
  //! construct such objects, but should use the default_value() function instead.
  //
  template <typename object_type, typename default_type>
  struct default_value_type {
    object_type& object;
    const default_type defaultvalue;
    default_value_type(object_type& object, const default_type defaultvalue)
      : object(object), defaultvalue(defaultvalue) { }
  };

  //! default_value() is used to read an object from a stream, assigning a
  //! default value if the read fails.  It returns a default_value_type{}
  //! object, which does the actual reading.
  //
  template <typename object_type, typename default_type>
  default_value_type<object_type,default_type>
  default_value(object_type& object, const default_type defaultvalue=default_type()) {
    return default_value_type<object_type,default_type>(object, defaultvalue);
  }

  //! This operator>>() reads default_value_type{} from an input stream.
  //
  template <typename object_type, typename default_type>
  std::istream& operator>> (std::istream& is, 
			    default_value_type<object_type, default_type> dv) {
    if (is) {
      if (is >> dv.object)
	;
      else {
	is.clear(is.rdstate() & ~std::ios::failbit);  // clear failbit
	dv.object = dv.defaultvalue;
      }
    }
    return is;
  }

}  // namespace util
