// utility.h
//
// Mark Johnson, 20th May 2000
//
// Defines loop macros foreach, cforeach
//  hash functions for pairs and vectors
//  input and output for pairs and vectors
//  dfind (default find function)
//  afind (find function, asserts key exists)

#ifndef UTILITY_H
#define UTILITY_H

#include <cassert>
#include <ext/slist>
#include <utility>
#include <vector>

// foreach is a simple loop construct
//
#define foreach(TYPE, VAR, STORE) \
   for (TYPE::iterator VAR = STORE.begin(); VAR != STORE.end(); ++VAR)

#define cforeach(TYPE, VAR, STORE) \
   for (TYPE::const_iterator VAR = STORE.begin(); VAR != STORE.end(); ++VAR)


namespace std {

// hash function for arbitrary pairs
//
template<class T1, class T2> struct hash<pair<T1,T2> > {
  size_t operator()(const pair<T1,T2>& p) const
  {
    return 2*hash<T1>()(p.first)+3*hash<T2>()(p.second);
  }
};


// hash function for slists
//
template<class T> struct hash<slist<T> > 
{ //  This is the fn hashpjw of Aho, Sethi and Ullman, p 436.
  size_t operator()(const slist<T>& s) const 
  {
    typedef vector<T>::const_iterator CI;

    unsigned long h = 0; 
    unsigned long g;
    CI p = s.begin();
    CI end = s.end();
      
    while (p!=end) {
      h = (h << 7) + hash<T>()(*p++);
      if ((g = h&0xf0000000)) {
 	h = h ^ (g >> 24);
	h = h ^ g;
      }}
    return size_t(h);
  }
};


// hash function for vectors
//
template<class T> struct hash<vector<T> > 
{ //  This is the fn hashpjw of Aho, Sethi and Ullman, p 436.
  size_t operator()(const vector<T>& s) const 
  {
    typedef vector<T>::const_iterator CI;

    unsigned long h = 0; 
    unsigned long g;
    CI p = s.begin();
    CI end = s.end();
      
    while (p!=end) {
      h = (h << 7) + hash<T>()(*p++);
      if ((g = h&0xf0000000)) {
 	h = h ^ (g >> 24);
	h = h ^ g;
      }}
    return size_t(h);
  }
};


// print pairs
//
template <class T1, class T2> 
ostream& operator<< (ostream& os, const pair<T1,T2>& p)
{
  return os << '<' << p.first << ',' << p.second << '>';
}


// print slist
//
template <class T>
ostream& operator<< (ostream& os, const slist<T>& s)
{
  os << "(";
  for (slist<T>::const_iterator i = s.begin(); i != s.end(); ++i) {
    if (i != s.begin())
      os << ", ";
    os << *i;
  }
  return os << ')';
}
 

// print vector
//
template <class T>
ostream& operator<< (ostream& os, const vector<T>& v)
{
  os << "(";
  for (size_t i = 0; i < v.size(); ++i) {
    os << v[i];
    if (i+1 != v.size())
      os << ", ";
  }
  return os << ")";
}
  

// read slist
//
template <class T>
istream& operator>> (istream& is, slist<T>& s)
{
  T e;

  while (is >> e) 
    s.push_front(e);

  return is;
}


// read vector
//
template <class T>
istream& operator>> (istream& is, vector<T>& v)
{
  T e;

  while (is >> e) 
    v.push_back(e);

  return is;
}

// dfind(Map, Key) returns the value associated with Key, or the
//  default value if no such Key exists
//
template <class Map, class Key>
inline typename Map::data_type dfind(Map& m, const Key& k)
{
  typename Map::iterator i = m.find(k);
  if (i == m.end())
    return typename Map::data_type();
  else
    return i->second;
}

template <class Map, class Key>
inline const typename Map::data_type dfind(const Map& m, const Key& k)
{
  typename Map::const_iterator i = m.find(k);
  if (i == m.end())
    return typename Map::data_type();
  else
    return i->second;
}


// afind(map, key) returns a reference to the value associated
//  with key in map.  It uses assert to check that the key's value
//  is defined.
//
template <class Map, class Key>
inline typename Map::data_type& afind(Map& m, const Key& k)
{
  typename Map::iterator i = m.find(k);
  assert(i != m.end());
  return i->second;
}

template <class Map, class Key>
inline const typename Map::data_type& afind(const Map& m, const Key& k)
{
  typename Map::const_iterator i = m.find(k);
  assert(i != m.end());
  return i->second;
}

} // namespace std

#endif
