// sym.h -- An abbreviated symbol table package
//
// (c) Mark Johnson, 10th March 2001
//
// Class sym implements a symbol table
//
// Symbols are disguised pointers to strings
//
// Constructors:
//
//  sym()                This returns an undefined symbol
//  sym(const string&)
//
// Conversions:
//
//  string(symbol)
//  symbol.defined()
//
// Global properties:
//
//  symbol::size()       The number of symbols defined
//  symbol::already_defined(const string&)

#ifndef SYM_H
#define SYM_H

#define __STL_NO_RELOPS_NAMESPACE

#include <cassert>
#include <ext/hash_set>
#include <iostream>
#include <string>
#include <utility>


namespace std
{

class symbol {
  
  struct eqstr
  {
    bool operator()(const string& s1, const string& s2) const
    {
      return s1==s2;
    }
  };
  
  struct hashstr //  This is the fn hashpjw of Aho, Sethi and Ullman, p 436.
  {
    size_t operator()(const string& s) const 
    {
      typedef string::const_iterator CI;
      
      unsigned long h = 0; 
      unsigned long g;
      CI p = s.begin();
      CI end = s.end();
      
      while (p!=end) {
	h = (h << 4) + (*p++);
	if ((g = h&0xf0000000)) {
	  h = h ^ (g >> 24);
	  h = h ^ g;
	}}
      return size_t(h);
    }
  };

  const string* sp;
  symbol(const string* sp_) : sp(sp_) { }

public:
  typedef hash_set<string, hashstr, eqstr> Table;
  static Table table;
  
  symbol() : sp(NULL) { }   // constructs an undefined symbol
  symbol(const string& s) : sp(&*(table.insert(s).first)) { };

  const string& string_reference() const { return *sp; }
  const string* string_pointer() const { return sp; }
  const char* c_str() const { return sp->c_str(); }

  bool defined() const { return sp != NULL; }

  static symbol undefined() { return symbol(NULL); }
  static size_t size() { return table.size(); }

  bool operator== (const symbol s) const { return sp == s.sp; }
  bool operator!= (const symbol s) const { return sp != s.sp; }
  bool operator< (const symbol s) const { return sp < s.sp; }
  bool operator<= (const symbol s) const { return sp <= s.sp; }
  bool operator> (const symbol s) const { return sp > s.sp; }
  bool operator>= (const symbol s) const { return sp >= s.sp; }
};

inline istream& operator>> (istream& is, symbol& s)
{
  string str;
  if (is >> str)
    s = symbol(str);
  return(is);
}

inline ostream& operator<< (ostream& os, const symbol& s)
{
  return os << s.string_reference();
}

template<> struct hash<symbol> {
  size_t operator()(symbol s) const
  {
    return size_t(s.string_pointer());
  }
};

}

#endif

  
