/* interaction.h
 *
 * (c) Mark Johnson, 27th March 2001
 *
 * float interaction(int n, float s[], float *ase)
 *
 * This routine returns information to determine if an n-way combination
 * of variables X_0=x_0, ..., X_n-1=x_n-1, occurs more frequently than the 
 * 0 ... n-1 way combinations of variables would lead us to expect.
 *
 * interaction() returns the MLE of the n-way interaction term in a
 * log-linear model of the data in s[], and ase is the asymptotic
 * standard error of estimate of this term.  This routine destructively
 * changes s[].
 *
 * s[] should be a vector of size 2^n.  If 0 <= m < n and the bits that
 * are turned on in m are i_1, ..., i_k, then s[m] contains the number
 * of times the combination of variables X_i_1=x_i_1, ..., X_i_k=x_i_k
 * was observed.  Thus s[0] is the total number of counts in the data,
 * s[1] is the number of times X_0=x_0 was observed in the data and
 * s[2^n-1] is the number of times X_0=x_0, ..., X_n-1=x_n-1 was observed
 * in the data.
 *
 * standard_errors() returns the number of standard errors that correspond
 * to a two-sided significance level sig.  This is a fairly slow calculation,
 * so you should avoid recalculating it if possible.
 *
 * You can construct a test of significance at level sig by determining if
 * interaction/ase > standard_errors.
 *
 * lower_lambda() returns the lower bound on the interaction term lambda.
 */

#ifndef INTERACTION_H
#define INTERACTION_H

#ifdef __cplusplus
extern "C" {
#endif

float interaction2(float s[4], float *ase);
float interaction(int n, float s[], float *ase);
float lower_lambda(int n, float s[], float alpha);

float standard_errors(float sig);
  
#ifdef __cplusplus
}
#endif

#endif /* INTERACTION_H */
