/* gammavariate.c -- generates samples from a Gamma distribution
 *
 * Mark Johnson, 1st May 2006
 */

/* uncomment to run a test routine for gammavariate */
/* #define GAMMATEST */

#include <assert.h>
#include <math.h> 

#include "gammavariate.h"
#include "mt19937ar.h"

/* This definition of gammavariate is from Python code in
 * the Python random module.
 */

double gammavariate(double alpha) {

  assert(alpha > 0);

  if (alpha > 1.0) {
    
    /* Uses R.C.H. Cheng, "The generation of Gamma variables with
       non-integral shape parameters", Applied Statistics, (1977), 26,
       No. 1, p71-74 */

    double ainv = sqrt(2.0 * alpha - 1.0);
    double bbb = alpha - log(4.0);
    double ccc = alpha + ainv;
    
    while (1) {
      double u1 = mt_genrand_real3();
      if (u1 > 1e-7  || u1 < 0.9999999) {
	double u2 = 1.0 - mt_genrand_real3();
	double v = log(u1/(1.0-u1))/ainv;
	double x = alpha*exp(v);
	double z = u1*u1*u2;
	double r = bbb+ccc*v-x;
	if (r + (1.0+log(4.5)) - 4.5*z >= 0.0 || r >= log(z))
	  return x;
      }
    }
  }
  else if (alpha == 1.0) {
    double u = mt_genrand_real3();
    while (u <= 1e-7)
      u = mt_genrand_real3();
    return -log(u);
  }
  else { 
    /* alpha is between 0 and 1 (exclusive) 
       Uses ALGORITHM GS of Statistical Computing - Kennedy & Gentle */
    
    while (1) {
      double u = mt_genrand_real3();
      double b = (exp(1) + alpha)/exp(1);
      double p = b*u;
      double x = (p <= 1.0) ? pow(p, 1.0/alpha) : -log((b-p)/alpha);
      double u1 = mt_genrand_real3();
      if (! (((p <= 1.0) && (u1 > exp(-x))) ||
	     ((p > 1.0)  &&  (u1 > pow(x, alpha - 1.0)))))
	return x;
    }
  }
}

#ifdef GAMMATEST
#include <stdio.h>

int main(int argc, char **argv) {
  int iteration, niterations = 1000000;

  for (iteration = 0; iteration < niterations; ++iteration) {
    double alpha = 100*mt_genrand_real3();
    double gv = gammavariate(alpha, 1);
    if (!finite(gv))
      fprintf(stderr, "iteration = %d, alpha = %lg, gammavariate(alpha) = %lg\n",
	      iteration, alpha, gv);
  }
  return 0;
}

#endif /* GAMMATEST */



#if 0

/*! gammavariate() returns samples from a Gamma distribution
 *! where alpha is the shape parameter and beta is the scale 
 *! parameter, using the algorithm described on p. 94 of 
 *! Gentle (1998) Random Number Generation and Monte Carlo Methods, 
 *! Springer.
 */

double gammavariate(double alpha) {

  assert(alpha > 0); 
  
  if (alpha > 1.0) {
    while (1) {
      double u1 = mt_genrand_real3();
      double u2 = mt_genrand_real3();
      double v = (alpha - 1/(6*alpha))*u1/(alpha-1)*u2;
      if (2*(u2-1)/(alpha-1) + v + 1/v <= 2 
         || 2*log(u2)/(alpha-1) - log(v) + v <= 1)
	return (alpha-1)*v;
    }
  } else if (alpha < 1.0) {  
    while (1) {
      double t = 0.07 + 0.75*sqrt(1-alpha);
      double b = alpha + exp(-t)*alpha/t;
      double u1 = mt_genrand_real3();
      double u2 = mt_genrand_real3();
      double v = b*u1;
      if (v <= 1) {
	double x = t*pow(v, 1/alpha);
	if (u2 <= (2 - x)/(2 + x))
	  return x;
	if (u2 <= exp(-x))
	  return x;
      }
      else {
	double x = log(t*(b-v)/alpha);
	double y = x/t;
	if (u2*(alpha + y*(1-alpha)) <= 1)
	  return x;
	if (u2 <= pow(y,alpha-1))
	  return x;
      }
    }
  }
  else  
    return -log(mt_genrand_real3());
} 


/*! gammavariate() returns a deviate distributed as a gamma
 *! distribution of order alpha, beta, i.e., a waiting time to the alpha'th
 *! event in a Poisson process of unit mean.
 *!
 *! Code from Numerical Recipes
 */

double nr_gammavariate(double ia) {
  int j;
  double am,e,s,v1,v2,x,y;
  assert(ia > 0);
  if (ia < 10) { 
    x=1.0; 
    for (j=1;j<=ia;j++) 
      x *= mt_genrand_real3();
    x = -log(x);
  } else { 
    do {
      do {
	do { 
	  v1=mt_genrand_real3();
	  v2=2.0*mt_genrand_real3()-1.0;
	} while (v1*v1+v2*v2 > 1.0); 
	y=v2/v1;
	am=ia-1;
	s=sqrt(2.0*am+1.0);
	x=s*y+am;
      } while (x <= 0.0);
      e=(1.0+y*y)*exp(am*log(x/am)-s*y);
    } while (mt_genrand_real3() > e);
  }
  return x;
} 

#endif
