
# require paste to do the form parsing
from paste.request import parse_formvars 
# require the Redland python binding
import RDF 
import sys 
 

class SparqlWSGIEndpoint():
    """Provide a WSGI application implementing a SPARQL endpoint"""
    
    
    def __init__(self, storage, default_graph_uri='', allow_default_graph=True, allow_from=True):
        """Initialise SPARQL endpoint for a Redland model"""
        self.storage = storage
        self.default_graph_uri = default_graph_uri
        self.allow_default_graph = allow_default_graph
        self.allow_from = allow_from
        
        # store models in a dictionary against their uris
        self.models = dict()


    def form(self, environ, start_response):
        """WSGI method: generate a view with a form to run a SPARQL
        query from our endpoint"""

        formtext = """<html><body>
     <form action="sparql" method='GET'>
        <textarea rows="20" cols="60" name="query"></textarea>
        %s
        <input type="submit"/>
     </form>
   </body></html>"""
   
        dg = 'Graph: <input type="text" name="default-graph-uri"/>'

        if self.allow_default_graph:
            formtext = formtext % (dg,)
        else:
            formtext = formtext % ("",)
        
        start_response("200 OK", [('Content-type', 'text/html')])
        return formtext
        
    def load(self, uri):
        """Load an RDF graph from the given URI into a new model.
        
        Returns the model."""
         
        if not self.models.has_key(uri):
            self.models[uri] = RDF.Model(self.storage)
            print >> sys.stderr, "loading graph from ", uri
            self.models[uri].load(uri)

        return self.models[uri]
        
    def add_model(self, uri, model, default=False):
        """Add an existing model to the endpoint indexed
        under it's URI. If default=True, make this the default
        graph."""
        
        self.models[uri] = model
        if default:
            self.default_graph_uri = uri
        
    def _query_has_from_clause(self, query):
        """Check whether this query has a FROM clause"""

        # use a regexp, could of course be fooled by some valid query
        
        import re
        if re.search("from\s(<\w+>|\w+:\w+)", query, re.IGNORECASE):
            return True
        else:
            return False
            
            
            
    def _query(self, environ, start_response):
        """Run the query, generate the WSGI response
        
        This could possibly be refactored to allow queries to be run
        by other RDF libraries (eg. RDFlib).  
        """
                    
        # don't like doing this again but if we don't we have to pass
        # too many things to this method
        fields = parse_formvars(environ)
        
        # check whether we have the right model, load if needed
        if self.allow_default_graph and \
           fields.has_key("default-graph-uri") and fields['default-graph-uri'] != '':    
            
           graphuri = fields['default-graph-uri']
           model = self.load(graphuri)
        else:
           model = self.load(self.default_graph_uri)

        # run the query
        try:
            q = RDF.Query(fields['query'], query_language="sparql")
            result = q.execute(model)
        except RDF.RedlandError, (value):
            start_response("400 Bad Request", [('Content-type', 'text/plain')])
            return "Query Error: "+str(value)
               
        # generate the result via WSGI
        if result.is_bindings() or result.is_boolean():
            start_response("200 OK", [('Content-type', 'application/sparql-results+xml')])
            return str(result)
        elif result.is_graph():
            start_response("200 OK", [('Content-type', 'application/rdf+xml')])
            return result.to_string()
                
                
                
    def process_query(self, environ, start_response):
        """Process the SPARQL query, return a WSGI response"""
        
        fields = parse_formvars(environ)
        if fields.has_key("query"): 
            
            if not self.allow_from and self._query_has_from_clause(fields['query']):
                start_response("500 Internal Server Error", [('Content-type', 'text/plain')])
                return "Query request refused: FROM clause in query not allowed on this server"

            return self._query(environ, start_response)

        else:
            # no query, return error
            start_response("400 Bad Request", [('Content-type', 'text/plain')])
            return "Bad Request: No query parameter sent"
    
    
    def __call__(self, environ, start_response):
        """Implement the WSGI endpoint callable"""
         
        return self.process_query(environ, start_response)
        
        

