// tree.h -- tree code
//
// (c) Mark Johnson, 12th February 2008
//
//! WARNING -- These trees DO NOT HAVE MEMORY MANAGEMENT.  It is up
//! to you to delete a tree if you want to reclaim the memory it uses!

#ifndef TREE_H
#define TREE_H

#include <cassert>
#include <cctype>
#include <iostream>
#include <sstream>
#include <string>

namespace tree {

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                          Utility                                      //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  template <typename Label=unsigned> struct bintree_type;   // forward declaration

  //! to_string(x) converts x to a string
  //
  template <typename X>
  std::string to_string(const X& x) {
    std::ostringstream os;
    os << x;
    return os.str();
  }

  const std::string& to_string(const std::string& x) {
    return x;
  }

  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                       Arbitrary branching trees                       //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! A tree_type{} is a node in an n-ary branching tree.
  //! It "owns" its child and next subtrees, i.e., if you delete a tree
  //! node then these are deleted also.
  //
  template <typename Label=unsigned>
  struct tree_type {
  
    typedef Label label_type;         //!< Type of label
    label_type label;                 //!< This node's label
    tree_type* subtrees;              //!< Pointer to this node's first child
    tree_type* next;                  //!< Pointer to this node's next sibling

    tree_type(Label label=Label(), tree_type* subtrees=NULL, tree_type* next=NULL)
      : label(label), subtrees(subtrees), next(next) 
    { }

    //! copy constructor copies this node and all its subtrees (i.e., a "deep" copy)
    //
    tree_type(const tree_type& t) 
      : label(t.label), 
	subtrees(t.subtrees ? new tree_type(t.subtrees) : NULL), 
	next(t.next ? new tree_type(t.next) : NULL) 
    { }

    //! = copies this node and all its subtrees (i.e., a "deep" copy)
    //
    tree_type& operator= (const tree_type& t) {
      if (this != &t) {
	label = t.label;
	delete subtrees;
	subtrees = (t.subtrees ? new tree_type(t.subtrees) : NULL);
	delete next;
	next = (t.next ? new tree_type(t.next) : NULL);
      }
      return *this;
    }  // tree::tree_type::operator=

    //! deleting a tree deletes its child and next subtrees as well
    //
    ~tree_type() {
      delete subtrees;
      delete next;
    }  // tree::tree_type::operator delete

    //! is_terminal() is true if this node has no children
    //
    bool is_terminal() const { return subtrees == NULL; }

    //! is_preterminal() is true if this node has a terminal child
    //
    bool is_preterminal() const { return subtrees != NULL && subtrees->subtrees == NULL; }

    //! is_nonterminal() is true if this node has a child, which in turn has a child
    //
    bool is_nonterminal() const { return subtrees != NULL && subtrees->subtrees != NULL; }

    //! size() returns the number of terminals in this tree
    //
    unsigned size() const {
      if (is_terminal())
	return 1;
      else
	return subtrees->size() + (next ? next->size() : 0);
    }  // tree::tree_type::size()

    //! yield(ws) pushes the terminals of this subtree onto the back of ws
    //
    template <typename Ws>
    void yield(Ws& ws) const {
      if (is_terminal())
	ws.push_back(label);
      else {
	if (subtrees)
	  subtrees->yield(ws);
      }
      if (next)
	next->yield(ws);
    }  // tree::tree_type::yield()

    //! preterminals(ps, ws) pushes the preterminals and terminals 
    //!  of this subtree onto the back of ps and ws respectively
    //
    template <typename Ps, typename Ws>
    void preterminals(Ps& ps, Ws& ws) const {
      if (is_preterminal()) {
	ps.push_back(label);
	ws.push_back(subtrees->label);
      }
      else {
	if (subtrees)
	  subtrees->preterminals(ps, ws);
      }
      if (next)
	next->preterminals(ps, ws);
    }  // tree::tree_type::preterminals()

    //! write() writes a tree onto an ostream in a single line.  
    //! It calls labelfn[label] to produce a label that will
    //! be written to the stream os.
    //
    template <typename Func>
    std::ostream& write(std::ostream& os, const Func& labelfn) const {
      if (is_terminal()) 
	os << labelfn[label];
      else {
	os << '(' << labelfn[label];
	for (tree_type* c = subtrees; c; c=c->next) {
	  os << ' ';
	  c->write(os, labelfn);
	}
	os << ')';
      }
      return os;
    }  // tree::tree_type::write()

    //! prettyprint() pretty-prints a tree across multiple lines.
    //! It calls labelfn[label] to produce a label that will be written
    //! to the stream os.
    //
    template <typename Func>
    std::ostream& prettyprint(std::ostream& os, const Func& labelfn, unsigned indent=0) const {
      if (is_terminal())
	os << labelfn[label];
      else {  // nonterminal
	std::string l = to_string(labelfn[label]);
	os << '(' << l << ' ';
	indent += l.size() + 2;     // for '(' and ' ';
	subtrees->prettyprint(os, labelfn, indent);
	for (tree_type* p = subtrees->next; p; p = p->next) {
	  os << '\n';
	  for (unsigned i = 0; i < indent; ++i)
	    os.put(' ');            // indent for next subtree
	  p->prettyprint(os, labelfn, indent);
	}
	os << ')';
      }
      return os;
    }  

    //! read() reads a tree from an istream.
    //! It calls string_label(s) to map a string s to a label.
    //
    template <typename Func>
    static tree_type* read(std::istream& is, Func& string_label) {
      int c;
      while (1)                        // skip spaces
	if ((c = is.get()) == EOF)
	  return NULL;
	else if (!isspace(c))
	  break;

      switch (c) {
      case ')':
	return NULL;                   // empty tree
	break;
      
      case '(':                        // full subtree
	while ((c = is.peek()) != EOF && isspace(c))
	  is.get();                    //  skip spaces
	{
	  std::string l;
	  while ((c = is.peek()) != EOF && !isspace(c) && c != '(' && c != ')') 
	    l.push_back(char(is.get()));
	  tree_type* t = new tree_type(string_label(l));
	  tree_type* p = t->subtrees = read(is, string_label);   // read child tree
	  while (p) 
	    p = p->next = read(is, string_label);
	  return t;
	}
	break;

      default:                          // terminal
	{
	  std::string l;
	  l.push_back(char(c));
	  while ((c = is.peek()) != EOF && !isspace(c) && c != '(' && c != ')') 
	    l.push_back(char(is.get()));
	  return new tree_type(string_label(l));
	}
	break;
      }
    }  // tree::tree_type::read()

    template <typename Func>
    bintree_type<Label>* bintree(Func& binlabel) const;  // forward declaration

  };  // tree::tree_type{}


  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                         Binary branching trees                        //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! A bintree_type{} is a node in a binary tree.
  //! It owns its first and second child trees, i.e., if you delete a bintree
  //! node then these are deleted also.
  //
  template <typename Label>
  struct bintree_type {

    typedef Label label_type;         //!< Type of label
    label_type label;                 //!< This node's label
    bintree_type* first;              //!< This node's first child
    bintree_type* second;             //!< This node's second child

    bintree_type(Label label=Label(), bintree_type first=NULL, bintree_type second=NULL)
      : label(label), first(first), second(second) { }

    // bintree_type(const bintree_type& b) : label(b.label), first(b.first), second(b.second) { }

    //! copy constructor copies this node and all its subtrees (i.e., a "deep" copy)
    //
    bintree_type(const bintree_type& b) 
      : label(b.label), 
	first(b.first ? new bintree_type(b.first) : NULL), 
	second(b.second ? new bintree_type(b.second) : NULL) 
    { }

    //! = copies this node and all its subtrees (i.e., a "deep" copy)
    //
    bintree_type& operator= (const bintree_type& b) {
      if (this != &b) {
	label = b.label;
	delete first;
	first = (b.first ? new bintree_type(b.first) : NULL);
	delete second;
	second = (b.second ? new bintree_type(b.second) : NULL);
      }
      return *this;
    }  // bintree_type::operator=

    //! deleting a bintree deletes its first and second subtrees
    //
    ~bintree_type() {
      delete first;
      delete second;
    }

    //! tree() maps this binary tree into an equivalent n-ary tree.
    //! It returns a pointer to the new tree.  The predicate keep determines
    //! which nodes appear in the n-ary tree, and which are collapsed.
    //! If keep(t) is true, then t is kept, otherwise t is not kept.
    //
    template <typename Pred>
    tree_type<Label>* tree(Pred keep, tree_type<Label>* sibling=NULL) const {
      if (!this) {
	assert(!sibling);  // if this node is NULL, the next one should be too
	return NULL;
      }
      else if (keep(this)) 
	return new tree_type<Label>(label, 
				    first->tree(keep, 
						second->tree(keep, NULL)), 
				    sibling);
      else
	return first->tree(keep, second->tree(keep, sibling));
    }  // bintree_type::tree()

    //! is_terminal() is true if this node has no children
    //
    bool is_terminal() const { return this != NULL && first == NULL; }

    //! is_preterminal() is true if this node has a terminal child
    //
    bool is_preterminal() const { return this != NULL && first != NULL && first->first == NULL; }

    //! is_nonterminal() is true if this node has a child, which in turn has a child
    //
    bool is_nonterminal() const { return this != NULL && first != NULL && first->first != NULL; }

  };  // bintree_type{}

  //! bintree() left-binarizes an n-ary tree.  
  //! binlabel() is a function object that constructs a new binary label
  //! given the labels on the children.  It will be called 
  //! binlabel(parentlabel,leftlabel,rightlabel) and should return the label
  //! for the new node.
  //
  template <typename Label> template <typename Func>
  bintree_type<Label>* tree_type<Label>::bintree(Func& binlabel) const {
    if (!this)
      return NULL;
    else if (subtrees == NULL)
      return new bintree_type<Label>(label);
    else if (subtrees->next == NULL)
      return new bintree_type<Label>(label, subtrees->bintree(binlabel));
    else {
      bintree_type<Label>* bsubtrees = subtrees->bintree(binlabel);
      tree_type* c;
      for (c = subtrees->next; c->next != NULL; c = c->next) 
	bsubtrees = new bintree_type<Label>(binlabel(label, bsubtrees->label, c->label),
					    bsubtrees,
					    c->bintree(binlabel));
      return new bintree_type<Label>(label, bsubtrees, c->bintree(binlabel));
    }
  }  // tree_type::bintree()


  ///////////////////////////////////////////////////////////////////////////
  //                                                                       //
  //                         Input and output routines                     //
  //                                                                       //
  ///////////////////////////////////////////////////////////////////////////

  //! An IdentityFn is a function object that computes the identity function.
  //! If you use strings as node labels, you'll probably want to use this to
  //! read in and print out treees.
  //
  struct IdentityFn  {
    template <typename X> const X& operator() (const X& x) const { return x; }
    template <typename X> const X& operator[] (const X& x) const { return x; }
  };  // tree::IdentityFn{}

  template <typename Label>
  std::istream& operator>> (std::istream& is, tree_type<Label>*& tp) {
    IdentityFn id;
    tp = tree_type<Label>::read(is, id);
    return is;
  }

  template <typename Label>
  std::ostream& operator<< (std::ostream& os, const tree_type<Label>* tp) {
    return tp->write(os, IdentityFn());
  }

  //! A TreeLabel is a pair consisting of a tree and a labeling object (e.g., 
  //!  a symtab{}).   It's purpose is to hold the labeling object and pass it
  //!  to the appropriate read and write routines.
  //
  template <typename Tree, typename Labeler>
  struct TreeLabeler {
    Tree*& treeptr;
    Labeler& labeler;
    TreeLabeler(Tree*& treeptr, Labeler& labeler) 
      : treeptr(treeptr), labeler(labeler) { }
  };

  template <typename Tree, typename Labeler>
  std::istream& operator>> (std::istream& is, TreeLabeler<Tree,Labeler> treelabeler) {
    treelabeler.treeptr = Tree::read(is, treelabeler.labeler);
    return is;
  }

  template <typename Tree, typename Labeler>
  std::ostream& operator<< (std::ostream& os, const TreeLabeler<Tree,Labeler>& treelabeler) {
    return treelabeler.treeptr->write(os, treelabeler.labeler);
  }

  //! Define operator, (the comma operator) to build a TreeLabeler from a 
  //!  tree pointer and a labeler.
  //
  template <typename Label, typename Labeler> 
  TreeLabeler<tree_type<Label>,Labeler> 
  operator, (tree_type<Label>*& treeptr, Labeler& labeler) {
    return TreeLabeler<tree_type<Label>, Labeler>(treeptr, labeler);
  }

}  // namespace tree

#endif // TREE_H
