// med.h -- Minimum edit distance 
//
// Mark Johnson, December 2002

#ifndef MED_H
#define MED_H

#include <algorithm>
#include <cassert>
#include <functional>
#include <iostream>
#include <vector>

// You need to provide a cost function that defines the cost
// of particular element alignments.

//! zero_one_cost defines a zero-one cost function object.
//! It is generic, in that it is defined for arbitraty types
//! of elements.  zero_one_cost()(e1,e2) returns 0 if e1 == e2
//! and returns 1 otherwise.
//
template <typename EltType>
class zero_one_cost : public std::binary_function<EltType,EltType,int> {
  const EltType emptyelt;
public:
  int operator() (const EltType& e1, const EltType& e2)
  {
    if (e1 == e2)
      return 0;
    else
      return 1;
  }
  zero_one_cost(const EltType& emptyelt = EltType()) : emptyelt(emptyelt) { }
  EltType empty() { return emptyelt; }
};


//! min_edit_distance() returns the minimum cost for aligning
//! the elements in the sequence x1s with the elements in the
//! sequence x2s with respect to costfn (which defaults to the
//! zero-one cost).
//
template <typename Es, typename CostFn>
typename CostFn::result_type min_edit_distance(const Es& x1s, const Es& x2s, 
					       CostFn costfn = zero_one_cost<typename Es::value_type>() )
{
  typedef typename CostFn::result_type Cost;
  typedef std::vector<Cost> Costs;

  int n1 = x1s.size();
  int n2 = x2s.size();
  int n21 = n2+1;

  Costs c((n1+1)*(n2+1));

  for (int i1 = n1-1; i1 >= 0; --i1)
    c[i1*n21+n2] = costfn(x1s[i1], costfn.empty())+ c[(i1+1)*n21+n2];
    // c[i1*n21+n2] = n1-i1;

  for (int i2 = n2-1; i2 >= 0; --i2)
    c[n1*n21+i2] = costfn(costfn.empty(), x2s[i2])+c[n1*n21+(i2+1)];
    // c[n1*n21+i2] = n2-i2;

  for (int i1 = n1-1; i1 >= 0; --i1)
    for (int i2 = n2-1; i2 >= 0; --i2) 
      c[i1*n21+i2] = std::min(costfn(x1s[i1], x2s[i2])+c[(i1+1)*n21+(i2+1)],
			      std::min(costfn(x1s[i1], costfn.empty())+c[(i1+1)*n21+i2],
				       costfn(costfn.empty(), x2s[i2])+c[i1*n21+(i2+1)]));
  return c[0];
}

//! min_edit_distance() returns the minimum cost for aligning
//! the sequence x1s with the sequence x2s with respect to
//! costfn, and sets y1s and y2s to a minimum cost alignment.
//! y1s and y2s will in general contain the elements of x1s
//! and x2s respectively with extra null elements inserted,
//! so that for each i, either y1s[i] is an element from x1s
//! and aligns with y2s[i], or else y1s[i] is a null element
//! produced by costfn.empty() inserted to make y1s and y2s
//! align.
//
template <typename Es, typename CostFn>
typename CostFn::result_type min_edit_distance(const Es& x1s, const Es& x2s,
					       Es& y1s, Es& y2s, 
					       CostFn costfn = zero_one_cost<typename Es::value_type>() )
{
  typedef typename CostFn::result_type Cost;
  typedef std::vector<Cost> Costs;

  int n1 = x1s.size();
  int n2 = x2s.size();
  int n21 = n2+1;

  Costs c((n1+1)*(n2+1), 0);

  for (int i1 = n1-1; i1 >= 0; --i1)
    c[i1*n21+n2] = costfn(x1s[i1], costfn.empty())+ c[(i1+1)*n21+n2];
    // c[i1*n21+n2] = n1-i1;

  for (int i2 = n2-1; i2 >= 0; --i2)
    c[n1*n21+i2] = costfn(costfn.empty(), x2s[i2])+c[n1*n21+(i2+1)];
    // c[n1*n21+i2] = n2-i2;

  for (int i1 = n1-1; i1 >= 0; --i1)
    for (int i2 = n2-1; i2 >= 0; --i2) 
      c[i1*n21+i2] = std::min(costfn(x1s[i1], x2s[i2])+c[(i1+1)*n21+(i2+1)],
			      std::min(costfn(x1s[i1], costfn.empty())+c[(i1+1)*n21+i2],
				       costfn(costfn.empty(), x2s[i2])+c[i1*n21+(i2+1)]));

  if (false) {   // write out table
    for (int i2 = 0; i2 < n2; ++i2)
      std::cout << '\t' << x2s[i2];
    std::cout << std::endl;

    for (int i1 = 0; i1 <= n1; ++i1) {
      if (i1 < n1)
	std::cout << x1s[i1];
      std::cout << ':';
      for (int i2 = 0; i2 <= n2; ++i2)
	std::cout << '\t' << c[i1*n21+i2];
      std::cout << std::endl;
    }
  }

  int i1 = 0;
  int i2 = 0;
  while (i1 < n1 || i2 < n2) {
    Cost c12 = c[i1*n21+i2];
    if (i1 < n1 && i2 < n2 &&
	c12 == costfn(x1s[i1], x2s[i2])+c[(i1+1)*n21+(i2+1)]) {
      y1s.push_back(x1s[i1++]);
      y2s.push_back(x2s[i2++]);
    }
    else if (i1 < n1 &&
	     c12 == costfn(x1s[i1], costfn.empty())+c[(i1+1)*n21+i2]) {
      y1s.push_back(x1s[i1++]);
      y2s.push_back(costfn.empty());
    }
    else {
      assert(i2 < n2);
      assert(c12 == costfn(costfn.empty(), x2s[i2])+c[i1*n21+(i2+1)]);
      y1s.push_back(costfn.empty());
      y2s.push_back(x2s[i2++]);
    }
  }

  assert(i1 == n1);
  assert(i2 == n2);

  return c[0];
}


#endif // MED_H
