// med.cc -- Minimum edit distance
//
// Mark Johnson

const char usage[] =
"med -- example progam using med.h, the minimum edit distance aligner\n"
"\n"
" (c) Mark Johnson, September 2003; last modified 29th September, 2011\n"
"\n"
"med reads pairs of strings from stdin, delimited by newline characters,\n"
"and writes out a pair of aligned strings, again delimited by newline\n"
"characters.  It continues reading pairs of lines until it encounters\n"
"end-of-file.\n"
"\n"
"First it aligns the words in the two strings.  Two words align if more\n"
"than half of their characters align.  Then it prints out the aligned\n"
"strings, indicating the word and character alignments.\n"
"\n"
"For example, given the two lines of input\n"
"\n"
"the big blue box\n"
"bigger red boxes\n"
"\n"
"the program produces as output\n"
"\n"
"the big    blue     box\n"
"    bigger      red boxes\n"
"\n"
"\nThis is actually just a demo program for med.h\n"
"which is a generic minimum edit distance aligner.\n";


#include <algorithm>
#include <functional>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>

#include "med.h"

typedef std::vector<std::string> strings;

template <typename T1, typename T2s>
void tokenize(const T1& e1, T2s& e2s)
{
  e2s.clear();
  std::istringstream is(e1);
  typedef typename T2s::value_type T2;
  T2 e2;
  while (is >> e2)
    e2s.push_back(e2);
}

class string_distance_cost : 
public std::binary_function<std::string, std::string,
			    typename std::string::size_type> {
public:
  int operator() (const std::string& s1, const std::string& s2) const
  {
    if (s1 == s2)
      return 0;
    else if (s1.empty())
      return std::max<result_type>(s2.size()/2, 1);
    else if (s2.empty())
      return std::max<result_type>(s1.size()/2, 1);
    else 
      return min_edit_distance(s1, s2, zero_one_cost<char>(' '));
  }
  std::string empty() const { return std::string(); }
};

int main(int argc, char** argv)
{
  if (argc != 1) {
    std::cerr << usage;
    exit(EXIT_FAILURE);
  }

  std::string line1, line2;

  while (getline(getline(std::cin, line1), line2)) {
    strings words1, words2, aligned1, aligned2;
    std::string out1, out2;
    tokenize(line1, words1);
    tokenize(line2, words2);
    min_edit_distance(words1, words2, aligned1, aligned2, 
		      string_distance_cost());
    assert(aligned1.size() == aligned2.size());
    for (size_t i = 0; i < aligned1.size(); ++i) {
      if (i > 0) {
	out1.push_back(' ');
	out2.push_back(' ');
      }
      min_edit_distance(aligned1[i], aligned2[i], out1, out2, 
			zero_one_cost<char>(' '));
    }
    std::cout << out1 << std::endl;
    std::cout << out2 << std::endl << std::endl;
  }
}
